﻿// ==============================================================================
// SUPPLEMENTARY AREAS PLUGIN
// ==============================================================================
	(async function ensureFluentReady() {
	if (window.fluentUIReady) {
		try {
			await window.fluentUIReady;
		} catch (awaitError) {
			console.warn("Supplementary Areas plugin: Fluent UI readiness promise rejected", awaitError);
		}
	}

	if (!window.fluent) {
		window.fluent = {};
	}

	if (!window.fluentUIReady) {
		window.fluentUIReady = Promise.resolve(true);
	}
	})();

(function(api, pluginId) {
	// ========================================
	// CONSTANTS & CONFIGURATION
	// ========================================

	/**
	 * Application constants
	 */
	const APP_CONSTANTS = {
		MAX_AREAS: 20,
		MAX_USERS_PER_FIELD: 50,
		FLASH_MESSAGE_TTL: 3500,
		SEARCH_DEBOUNCE_DELAY: 250,
		DEFAULT_NOTIFICATION_TYPE: 'info'
	};
	
	/**
	 * Notification types
	 * @readonly
	 * @enum {string}
	 */
	const NOTIFICATION_TYPES = {
		INFO: 'info',
		SUCCESS: 'success',
		ERROR: 'error'
	};
	/**
	 * Supplementary Areas Tab Web Component
	 * Manages supplementary areas for briefing records with full CRUD operations
	 */
	class SupplementaryAreasTab extends HTMLElement {
		/** @type {string[]} */
		static get observedAttributes() {
			return ['data-props'];
		}

		/**
		 * Initialize the component
		 */
		constructor() {
			super();
			this.attachShadow({ mode: 'open' });

			// ========================================
			// STYLES - Injected once in constructor
			// ========================================
			this._injectStyles();

			// ========================================
			// COMPONENT STATE
			// ========================================

			/** @type {Object} Component properties parsed from data-props attribute */
			this._props = {};

			/** @type {?Object} Main model containing dueDate, areas array, and version */
			this._model = null;

			/** @type {Array} Required field metadata from backend */
			this._fields = [];

			/** @type {?Object} Record summary including canUserEditMetadata */
			this._record = null;

			// ========================================
			// LOADING & ERROR STATE
			// ========================================

			/** @type {boolean} Whether component is currently loading data */
			this._loading = false;

			/** @type {boolean} Whether component is currently saving data */
			this._saving = false;

			/** @type {?string} Current error message to display */
			this._error = null;

			// ========================================
			// UI STATE
			// ========================================

			/** @type {Set<string>} Set of expanded area IDs for UI state */
			this._expanded = new Set();

			/** @type {?Object} Flash notification message state */
			this._flashMessage = null;

			/** @type {?number} Timer ID for flash message auto-dismissal */
			this._flashTimer = null;

			// ========================================
			// DATA STATE
			// ========================================

			/** @type {boolean} Whether there are unsaved changes */
			this._dirtyValues = false;

			/** @type {?string} Hash snapshot of last saved/loaded value state */
			this._baselineValuesHash = null;

			/** @type {?Object} Dropdown options for area names */
			this._dropdownOptions = [];

			/** @type {?Object} Missing fields configuration */
			this._missingFields = { global: [], perArea: {} };

			/** @type {?string} Internal name of the due date field (resolved server-side or via heuristic) */
			this._dueDateFieldInternalName = null;

			// Maximum number of users per field is defined in APP_CONSTANTS

			// ========================================
			// EVENT LISTENERS & CLEANUP
			// ========================================

			/** @type {Set<Function>} Collection of cleanup functions for event listeners */
			this._cleanupFunctions = new Set();
		}

		/**
		 * Inject styles into shadow DOM (called once in constructor)
		 * @private
		 */
		_injectStyles() {
			
			// inject loader styles for better performance
			const inlineStyleElement = document.createElement('style');
			inlineStyleElement.textContent = `
				:host {
				  display: block;
				  font-family: Segoe UI, system-ui, sans-serif;
				  font-size: 12px;
				  color: #323130;
				}
				
				.loading-container {
				  display: flex;
				  flex-direction: column;
				  align-items: center;
				  justify-content: center;
				  padding: 60px 20px;
				  gap: 16px;
				}
				
				.loading-text {
				  color: #0052c2;
				  font-size: 12px;
				}
			`;
			this.shadowRoot.appendChild(inlineStyleElement);

			const linkEl = document.createElement('link');
			linkEl.rel = "stylesheet";
			linkEl.href = `${window.pluginsResourcesBaseUrl}/${pluginId}/styles.css`;
			this.shadowRoot.appendChild(linkEl);
		}

		/**
		 * Handle attribute changes
		 * @param {string} name - Attribute name
		 * @param {string} oldValue - Previous value
		 * @param {string} newValue - New value
		 */
		attributeChangedCallback(name, oldValue, newValue) {
			if (name === 'data-props' && oldValue !== newValue) {
				this._parseProps();
				this._loadInitialData();
			}
		}

		/**
		 * Handle component connection to DOM
		 */
		connectedCallback() {
			this._parseProps();
			this._loadInitialData();
		}
		
		/**
		 * Apply Fluent UI accent color theme to shadow DOM container
		 * @private
		 */
		_applyTheme() {
			const provider = this.shadowRoot?.getElementById('theme-root');
			if (!provider || !window.fluent) return;
			
			const {
				accentBaseColor,
				accentFillRest, 
				accentFillHover, 
				accentFillActive,
				accentForegroundRest, 
				accentForegroundHover, 
				accentForegroundActive,
				controlCornerRadius,
				layerCornerRadius,
				SwatchRGB
			} = window.fluent;
			
			// Accent color
			const customAccentColor = SwatchRGB.from({ r: 0, g: 0.32, b: 0.76 });
		  	accentBaseColor.setValueFor(provider, customAccentColor);
			accentFillRest.setValueFor(provider, customAccentColor);
			accentFillActive.setValueFor(provider, customAccentColor);
			accentForegroundRest.setValueFor(provider, customAccentColor);
			accentForegroundActive.setValueFor(provider, customAccentColor);
			controlCornerRadius.setValueFor(provider, 2);
			layerCornerRadius.setValueFor(provider, 0);
		}

		/**
		 * Handle component disconnection from DOM
		 */
		disconnectedCallback() {
			this._cleanup();
		}

		/**
		 * Parse component properties from data-props attribute
		 * @private
		 */
		_parseProps() {
			try {
				this._props = JSON.parse(this.getAttribute('data-props') || '{}');
			} catch (error) {
				this._handleError('Failed to parse component properties', error);
				this._props = {};
			}
		}

		// ========================================
		// PUBLIC API METHODS
		// ========================================

		/**
		 * Check if the current user can edit metadata
		 * @returns {boolean} True if user can edit
		 */
		canEdit() {
			return !!(this._record && this._record.canUserEditMetadata);
		}

		// ========================================
		// DATA LOADING & INITIALIZATION
		// ========================================

		/**
		 * Reload data from the backend (resets all unsaved changes)
		 * @returns {Promise<void>}
		 */
		async _reloadData() {
			this._expanded.clear(); // Reset expanded state
			await this._loadInitialData();
		}

		/**
		 * Load initial data from the backend
		 * @private
		 * @returns {Promise<void>}
		 */
		async _loadInitialData() {
			if (this._loading) return;

			const recordId = this._props?.record?.recordId;
			if (!recordId) {
				this._render();
				return;
			}

			this._loading = true;
			this._error = null;
			this._render();

			try {
				const response = await api.invokeGet(undefined, { recordId });

				if (response.status !== 200) {
					throw new Error(`Load failed (${response.status})`);
				}

				await this._processLoadedData(response.body);
			} catch (error) {
				this._handleError('Failed to load data', error);
			} finally {
				this._loading = false;
				this._render();
			}
		}

		/**
		 * Process and validate loaded data from backend
		 * @private
		 * @param {string} responseBody - Raw response body
		 * @returns {Promise<void>}
		 */
		async _processLoadedData(responseBody) {
			try {
				const data = JSON.parse(responseBody);

				// Validate required data structure
				if (!data || typeof data !== 'object') {
					throw new Error('Invalid response structure');
				}

			this._model = data.model;
			this._record = data.record || this._record;
			this._fields = data.fields || [];
			// Accept explicit due date field internal name from server
			if (data.dueDateFieldInternalName && typeof data.dueDateFieldInternalName === 'string') {
				this._dueDateFieldInternalName = data.dueDateFieldInternalName;
			}
		this._dropdownOptions = data.dropdownOptions || [];

		this._processMissingFields(data.missingFields);
		this._normalizeAreaIndices();
			
			// Model now contains all data - no need to hydrate from fieldValues

				// Ensure at least one area exists
				if (this._model && Array.isArray(this._model.areas)) {
					if (this._model.areas.length === 0) {
					this._ensureInitialArea();
					}
				}

				// Establish baseline for change tracking
				this._updateBaselineHash();
				this._dirtyValues = false;

			} catch (error) {
				if (error instanceof SyntaxError) {
					this._handleError('Invalid JSON response');
				} else {
					this._handleError('Failed to process loaded data', error);
				}
			}
		}

		/**
	 * Process missing fields configuration from backend
		 * @private
		 * @param {Object|Array|undefined} missingFields - Missing fields data
		 */
		_processMissingFields(missingFields) {
		// Backend now returns { global: [...] } with aggregated field names
			if (!missingFields) {
			this._missingFields = { global: [] };
				return;
			}

		// Handle array format (legacy or simplified)
			if (Array.isArray(missingFields)) {
			this._missingFields = { global: missingFields };
				return;
			}

		// Handle object format with global property
		if (typeof missingFields === 'object' && 'global' in missingFields) {
				this._missingFields = {
				global: Array.isArray(missingFields.global) ? missingFields.global : []
				};
				return;
			}

		// Default to empty
		this._missingFields = { global: [] };
	}


		// ========================================
		// AREA MANAGEMENT METHODS
		// ========================================

	/**
	 * Toggle area expansion state
	 * @param {string} areaId - ID of the area to toggle
	 */
	toggleArea(areaId) {
		// Fluent accordion manages its own expanded state
		// We just track it for our internal state
		const panel = this.shadowRoot?.getElementById(`panel-${areaId}`);
		if (panel) {
			panel.expanded = !panel.expanded;
			if (panel.expanded) {
				this._expanded.add(areaId);
			} else {
				this._expanded.delete(areaId);
			}
		}
	}

		/**
		 * Add a new supplementary area
		 * @returns {Promise<void>}
		 */
		async addArea() {
			if (!this.canEdit() || !this._model) {
				this._notify('Cannot add area: insufficient permissions or no model loaded', NOTIFICATION_TYPES.ERROR);
				return;
			}

			if (this._model.areas.length >= APP_CONSTANTS.MAX_AREAS) {
				this._notify(`Maximum ${APP_CONSTANTS.MAX_AREAS} areas allowed`, NOTIFICATION_TYPES.ERROR);
				return;
			}

			const nextIndex = this._getNextAreaIndex();
			const newArea = this._createNewArea(nextIndex);

			this._model.areas.push(newArea);
		this._markAsDirty();
				this._render();
		}

		/**
		 * Delete a supplementary area
		 * @param {Object} area - Area object to delete
		 * @returns {Promise<void>}
		 */
		async deleteArea(area) {
			if (!this.canEdit() || !this._model) {
				this._notify('Cannot delete area: insufficient permissions or no model loaded', NOTIFICATION_TYPES.ERROR);
				return;
			}

			if (!confirm('Delete this supplementary area?')) {
				return;
			}

			this._model.areas = this._model.areas.filter(a => a.id !== area.id);
			this._expanded.delete(area.id);
		this._markAsDirty();
				this._render();
		}

	/**
	 * Update an area with new values
	 * @param {Object} area - Area object to update
	 * @param {Object} updates - Updates to apply
	 */
	updateArea(area, updates) {
		if (!this.canEdit() || !this._model) {
			return;
		}

		// Validate updates object
		if (!updates || typeof updates !== 'object') {
			console.warn('Invalid updates object provided to updateArea');
			return;
		}

		// Apply updates and mark as dirty for later persistence
		Object.assign(area, updates);
		this._markAsDirty();
		
		// If updating people fields, refresh the badges display
		const peopleFields = ['executiveDirectors', 'allocators', 'contributors'];
		const updatedPeopleFields = Object.keys(updates).filter(key => peopleFields.includes(key));
		if (updatedPeopleFields.length > 0) {
			updatedPeopleFields.forEach(fieldName => {
				this._refreshPeopleBadges(area, fieldName);
			});
		}
		// Don't re-render entire component to avoid UI jumping and accordion collapsing
	}

	/**
	 * Update the due date
	 * @param {string|null} value - New due date value
	 */
	updateDueDate(value) {
		if (!this.canEdit() || !this._model) {
			return;
		}

		this._model.dueDate = value || null;
		this._markAsDirty();
		// Don't re-render to avoid UI jumping
	}


		// ========================================
		// PERSISTENCE METHODS
		// ========================================

	/**
	 * Save complete state to server (all changes in one call)
	 * @returns {Promise<void>}
	 */
	async saveAll() {
		if (!this.canEdit() || !this._model) return;
		
		// IMPORTANT: Build payload BEFORE re-rendering to preserve input values
		// Re-rendering destroys and recreates DOM elements, losing user input
		const payload = this._buildCompleteStatePayload();
		
		this._saving = true;
		this._render();
		
		try {
			const updatedModel = await this._saveCompleteState(payload);
			
			if (updatedModel) {
				this._model = updatedModel;
				this._normalizeAreaIndices();
				this._updateBaselineHash();
				this._dirtyValues = false;
			} else {
				// Fallback: reload if backend didn't return model
				await this._loadInitialData();
			}
			
			this._notify('All changes saved successfully', NOTIFICATION_TYPES.SUCCESS);
		} catch (err) {
			this._handleError('Failed to save', err);
		} finally {
			this._saving = false;
			this._render();
		}
	}

	/**
	 * Build complete state payload for saving
	 * @private
	 * @returns {Object} Complete state payload
	 */
	_buildCompleteStatePayload() {
		return {
			version: this._model.version,
			dueDate: this._model.dueDate || null,
			recordType: this._props.record.recordType, // Required by backend to identify missing fields
			areas: (this._model.areas || []).map(area => {
				// Check if there's a new comment in the input field
				const commentInput = this.shadowRoot?.getElementById(`new-comment-${area.id}`);
				const newCommentText = commentInput ? (commentInput.currentValue || commentInput.value || '').trim() : null;

				return {
					id: area.id,
					label: area.label || null,
					order: typeof area.order === 'number' ? area.order : (area.index || 0),
					index: typeof area.index === 'number' ? area.index : (area.order || 0),
					executiveDirectors: area.executiveDirectors || [],
					allocators: area.allocators || [],
					contributors: area.contributors || [],
					comments: area.comments || [],
					newComment: newCommentText || null,
					endorsementComplete: !!area.endorsementComplete
				};
			})
		};
	}

	/**
	 * Save complete state to the backend
	 * @private
	 * @param {Object} payload - Complete state payload
	 * @returns {Promise<Object>} Updated model from server
	 */
	async _saveCompleteState(payload) {
		const body = JSON.stringify(payload);

		const response = await api.invokePost('updateRecord', {
			recordId: this._props.record.recordId,
			recordType: this._props.record.recordType // Pass as query param for backend compatibility
		}, body);

		if (response.status !== 200) {
			throw new Error(`Save failed: ${response.status}`);
		}

		const data = JSON.parse(response.body);
		
		// Backend now returns the full updated model (including new comments)
		if (data.model) {
			return data.model;
		}
		
		// Fallback for legacy response format (just modelVersion)
		if (data.modelVersion) {
			this._model.version = data.modelVersion;
		}
		
		return null;
	}
		

		// ========================================
		// HELPER METHODS
		// ========================================

		/**
		 * Get the next available area index
		 * @private
		 * @returns {number} Next available index
		 */
		_getNextAreaIndex() {
			if (!this._model || !Array.isArray(this._model.areas) || this._model.areas.length === 0) {
				return 1;
			}

			const maxIndex = Math.max(...this._model.areas.map(area =>
				typeof area.index === 'number' ? area.index : (typeof area.order === 'number' ? area.order : 0)
			));

			return maxIndex + 1;
		}

		/**
		 * Create a new area object
		 * @private
		 * @param {number} index - Area index
		 * @returns {Object} New area object
		 */
		_createNewArea(index) {
			const uuid = crypto.randomUUID ? crypto.randomUUID() : (`id${Date.now()}${Math.random()}`);

			return {
				id: uuid,
				index,
				order: index,
				label: null,
				executiveDirectors: [],
				allocators: [],
				contributors: [],
				comments: [],
				endorsementComplete: false
			};
		}

		/**
		 * Ensure at least one initial area exists
		 * @private
		 */
	_ensureInitialArea() {
			if (this._model.areas.length === 0) {
				const initialArea = this._createNewArea(1);
				this._model.areas.push(initialArea);
			this._markAsDirty();
		}
	}


		// ========================================
		// NOTIFICATION METHODS
		// ========================================

		/**
		 * Show a notification message to the user
		 * @param {string} message - Message to display
		 * @param {string} type - Type of notification (info, success, error)
		 * @param {number} [duration=3500] - Duration in milliseconds
		 */
		_notify(message, type = NOTIFICATION_TYPES.INFO, duration = APP_CONSTANTS.FLASH_MESSAGE_TTL) {
			this._flashMessage = {
				text: message,
				type,
				expires: Date.now() + duration
			};

			if (this._flashTimer) {
				clearTimeout(this._flashTimer);
			}

			this._flashTimer = setTimeout(() => {
				this._flashMessage = null;
				this._flashTimer = null;
				this._render();
			}, duration);

			// Only trigger re-render if not already saving/loading
			if (!this._loading && !this._saving) {
				this._render();
			}
		}

		// ========================================
		// RENDERING METHODS
		// ========================================

	/**
	 * Render the component to the DOM
	 * @private
	 */
	_render() {
		const content = this._renderContent();
		
		// Check if container exists, create if first render
		let container = this.shadowRoot.getElementById('theme-root');
		if (!container) {
			container = document.createElement('div');
			container.id = 'theme-root';
			this.shadowRoot.appendChild(container);
		}
		
		// Update content without destroying styles
		container.innerHTML = content;
		this._attachEventListeners();
		this._applyTheme();
	}

	/**
	 * Render flash notification message
	 * @private
	 * @returns {string} HTML for flash message
	 */
	_renderFlashMessage() {
		if (!this._flashMessage) return '';

		// Map notification type to visual styling
		const icon = this._flashMessage.type === 'success' ? '✓' : 
		             this._flashMessage.type === 'error' ? '✕' : 'ℹ';
		const className = `notification-card notification-flash notification-${this._flashMessage.type}`;
		
		return `<fluent-card class="${className}" role="status" aria-live="polite">
			<div class="notification-content">
				<span class="notification-icon">${icon}</span>
				<span class="notification-text">${this._escapeHtml(this._flashMessage.text)}</span>
			</div>
		</fluent-card>`;
	}

	/**
	 * Render main component content
	 * @private
	 * @returns {string} HTML content
	 */
	_renderContent() {
		if (this._loading) {
			return '<div class="loading-container"><fluent-progress-ring></fluent-progress-ring><div class="loading-text">Loading supplementary areas...</div></div>';
		}

			if (this._error) {
				return `<div style="color:#b00020">${this._escapeHtml(this._error)}</div>`;
			}

			if (!this._model) {
				return '<div>No data.</div>';
			}

		const sortedAreas = this._getSortedAreas();
		const areasHtml = sortedAreas.map(area => this._renderArea(area)).join('');

	// Check for global missing fields (aggregated fields that don't exist in content type)
		const globalMissing = (this._missingFields && Array.isArray(this._missingFields.global)) ? this._missingFields.global : [];
	const globalMissingHtml = this._renderGlobalMissingFields(globalMissing);

	// Due date field internal name is set from server on initial load
	const dueDateField = (this._fields || []).find(field => field.internalName === this._dueDateFieldInternalName);
		const dueDateInternal = this._dueDateFieldInternalName;
		const dueDateMissing = (!!dueDateInternal) && ((dueDateField && dueDateField.isMissing) || (!dueDateField && globalMissing.includes(dueDateInternal)));
	
		let dueDateSection;
		if(dueDateMissing){
		// Due date is missing - show inline notice
		dueDateSection = `<div class="field-wrapper"><label>Due Date</label><div style="color:#b00020;font-size:11px;">Field "${dueDateInternal}" missing from content type</div></div>`;
		} else {
			// Extract date and time parts from ISO 8601 format
			const dateValue = this._extractDatePart(this._model.dueDate || '');
			const timeValue = this._extractTimePart(this._model.dueDate || '');
			const dueDateFluentInput = `<div class="due-date-time-wrapper">
				<fluent-text-field appearance="outline" type="date" id="dueDateInput" value="${this._escapeAttr(dateValue)}" ${!this.canEdit() ? 'disabled' : ''}></fluent-text-field>
				<fluent-select appearance="outline" id="dueTimeInput" ${!this.canEdit() ? 'disabled' : ''} value="${this._escapeAttr(timeValue)}">${this._renderTimeOptions(timeValue)}</fluent-select>
			</div>`;
			dueDateSection = `<div class="field-wrapper"><label>Due Date</label>${this._renderField(dueDateField, dueDateFluentInput)}</div>`;
		}

	return `
		<div class="form-container">
			${this._renderFlashMessage()}
			${globalMissingHtml}
		<div class="top-section">
			${dueDateSection}
			${this.canEdit() ? `<fluent-button id="addBtn" appearance="accent" ${this._saving || this._model.areas.length >= APP_CONSTANTS.MAX_AREAS ? 'disabled' : ''} title="${this._saving ? 'Cannot add areas while saving' : ''}">
				New supplementary area
			</fluent-button>` : ''}
		</div>
			<fluent-accordion class="areas" expand-mode="multi">
				${areasHtml}
			</fluent-accordion>
		<div class="bottom-toolbar">
			${this.canEdit() ? `<fluent-button id="saveBtn" appearance="accent" ${this._saving || !this._dirtyValues ? 'disabled' : ''}>
				${this._saving ? 'Saving...' : 'Save'}
			</fluent-button>` : ''}
			<fluent-button id="cancelBtn" appearance="outline" ${this._saving ? 'disabled' : ''}>
				Cancel
			</fluent-button>
		</div>
		</div>
	`;
	}

	/**
	 * Attach event listeners to DOM elements
	 * @private
	 */
	_attachEventListeners() {
		this._cleanup(); // Clean up previous listeners

		if (!this._model) return;

		// Due date input
		const dueDateInput = this.shadowRoot.getElementById('dueDateInput');
		const dueTimeInput = this.shadowRoot.getElementById('dueTimeInput');
		
		const updateDueDateWithTime = () => {
			const datePart = dueDateInput ? dueDateInput.value : '';
			const timePart = dueTimeInput ? dueTimeInput.value : '';
			const combined = this._combineDateAndTime(datePart, timePart);
			this.updateDueDate(combined);
		};
		
		if(dueDateInput){
			const handler = () => updateDueDateWithTime();
			dueDateInput.addEventListener('change', handler);
			this._cleanupFunctions.add(()=> dueDateInput.removeEventListener('change', handler));
		}
		
		if(dueTimeInput){
			const handler = () => updateDueDateWithTime();
			dueTimeInput.addEventListener('change', handler);
			this._cleanupFunctions.add(()=> dueTimeInput.removeEventListener('change', handler));
		}

		// Add area button
		const addBtn = this.shadowRoot.getElementById('addBtn');
		if(addBtn){
			const handler = ()=> this.addArea();
			addBtn.addEventListener('click', handler);
			this._cleanupFunctions.add(()=> addBtn.removeEventListener('click', handler));
		}

	// Save button
	const saveBtn = this.shadowRoot.getElementById('saveBtn');
	if(saveBtn){
		const handler = ()=> this.saveAll();
		saveBtn.addEventListener('click', handler);
		this._cleanupFunctions.add(()=> saveBtn.removeEventListener('click', handler));
		}

		// Cancel button - reload data to reset all unsaved changes
		const cancelBtn = this.shadowRoot.getElementById('cancelBtn');
		if(cancelBtn){
			const handler = ()=> this._reloadData();
			cancelBtn.addEventListener('click', handler);
			this._cleanupFunctions.add(()=> cancelBtn.removeEventListener('click', handler));
		}

		// Area-specific listeners
		this._model.areas.forEach(area=> this._attachAreaEventListeners(area));
	}

	/**
	 * Attach event listeners for a specific area
	 * @private
	 * @param {Object} area - Area object
	 */
	_attachAreaEventListeners(area) {
		// Fluent accordion item handles expand/collapse via 'change' event
		const panel = this.shadowRoot.getElementById(`panel-${area.id}`);
		if(panel){
			const handler = (e)=> {
				// Sync our expanded state with the accordion's state
				if(e.target.expanded) {
					this._expanded.add(area.id);
				} else {
					this._expanded.delete(area.id);
				}
			};
			panel.addEventListener('change', handler);
			this._cleanupFunctions.add(()=> panel.removeEventListener('change', handler));
		}
	if(!this.canEdit()) return;
	const deleteLink = this.shadowRoot.getElementById(`del-${area.id}`);
	if(deleteLink){
		const handler = (e)=> {
			e.stopPropagation(); // Prevent accordion toggle
			this.deleteArea(area);
		};
		deleteLink.addEventListener('click', handler);
		this._cleanupFunctions.add(()=> deleteLink.removeEventListener('click', handler));
	}
			
		// Select component
		const nameSelect = this.shadowRoot.getElementById(`name-${area.id}`);
		if(nameSelect){
			const handler = (e)=> {
				// For fluent-select, get the selected value
				const selectedValue = e.target.value;
				// Find the label for this value from dropdown options
				const option = (this._dropdownOptions || []).find(opt => 
					(opt.id === selectedValue) || (opt.value === selectedValue)
				);
				const label = option ? (option.label || option.id || option.value) : selectedValue;
				this.updateArea(area, { label: label || null });
				
				// Update endorsement checkbox state based on whether area name is selected
				const endorseCheckbox = this.shadowRoot.getElementById(`endorse-${area.id}`);
				if (endorseCheckbox) {
					const hasAreaName = !!(label && label.trim());
					endorseCheckbox.disabled = !this.canEdit() || !hasAreaName;
					// If unchecking the name and endorsement was checked, uncheck endorsement
					if (!hasAreaName && area.endorsementComplete) {
						area.endorsementComplete = false;
						endorseCheckbox.checked = false;
						this._markAsDirty();
					}
				}
				
				// Update the heading to show/hide endorsement indicator
				this._updateAreaHeading(area);
			};
			nameSelect.addEventListener('change', handler);
			this._cleanupFunctions.add(()=> nameSelect.removeEventListener('change', handler));
		}
	// Checkbox component
		const endorseCheckbox = this.shadowRoot.getElementById(`endorse-${area.id}`);
		if(endorseCheckbox){
			const handler = e=> {
				this.updateArea(area, { endorsementComplete: e.target.checked || e.target.currentChecked });
				// Update the heading to show/hide endorsement indicator
				this._updateAreaHeading(area);
			};
			endorseCheckbox.addEventListener('change', handler);
			this._cleanupFunctions.add(()=> endorseCheckbox.removeEventListener('change', handler));
		}

	// New comment input - mark dirty on input
	const commentInput = this.shadowRoot.getElementById(`new-comment-${area.id}`);
	if(commentInput){
		const handler = ()=> this._markAsDirty();
		commentInput.addEventListener('input', handler);
		this._cleanupFunctions.add(()=> commentInput.removeEventListener('input', handler));
	}

		this._attachPeoplePickerListeners(area);
	}

	/**
	 * Refresh the people badges display for a specific field without full re-render
	 * @private
	 * @param {Object} area - Area object
	 * @param {string} fieldName - Field name (executiveDirectors, allocators, contributors)
	 */
	_refreshPeopleBadges(area, fieldName) {
		const container = this.shadowRoot.getElementById(`pp-${fieldName}-${area.id}`);
		if (!container) return;

		const badgesContainer = container.querySelector('.people-badges-container');
		const countElement = container.querySelector('.people-count');
		const input = container.querySelector('.people-search-input');

		if (!badgesContainer || !countElement) return;

		const people = Array.isArray(area[fieldName]) ? area[fieldName] : [];
		const isAtLimit = people.length >= APP_CONSTANTS.MAX_USERS_PER_FIELD;

		// Update badges with delete button inside
		const badgesHtml = people.map(p => {
			const key = this._getPersonKey(p);
			const label = this._getPersonDisplay(p);
			return `<fluent-badge appearance="accent" class="person-badge" data-person-key="${this._escapeAttr(key)}">
				${this._escapeHtml(label)}
				${this.canEdit() ? `<button class="badge-remove" data-key="${this._escapeAttr(key)}" aria-label="Remove ${this._escapeAttr(label)}" title="Remove ${this._escapeAttr(label)}" type="button">×</button>` : ''}
			</fluent-badge>`;
		}).join('');
		badgesContainer.innerHTML = badgesHtml;

		// Update count
		countElement.textContent = `${people.length}/${APP_CONSTANTS.MAX_USERS_PER_FIELD}`;
		countElement.setAttribute('aria-label', `${people.length} of ${APP_CONSTANTS.MAX_USERS_PER_FIELD} selected`);

		// Update input state
		if (input) {
			if (isAtLimit) {
				input.disabled = true;
				input.placeholder = `Maximum ${APP_CONSTANTS.MAX_USERS_PER_FIELD} users reached`;
			} else {
				input.disabled = false;
				input.placeholder = 'Type to search people...';
			}
		}

		// Update container class
		if (isAtLimit) {
			container.classList.add('at-limit');
		} else {
			container.classList.remove('at-limit');
		}

		// Re-attach remove button event listeners
		Array.from(badgesContainer.querySelectorAll('button.badge-remove')).forEach(btn => {
			const removeHandler = (ev) => {
				if(ev && ev.type==='keydown' && ev.key!=='Enter' && ev.key!==' ') return;
				const key = btn.getAttribute('data-key');
				const updated = (area[fieldName] || []).filter(p => this._getPersonKey(p) !== key);
				this.updateArea(area, { [fieldName]: updated });
			};
			btn.addEventListener('click', removeHandler);
			btn.addEventListener('keydown', removeHandler);
			this._cleanupFunctions.add(()=>{ btn.removeEventListener('click', removeHandler); btn.removeEventListener('keydown', removeHandler); });
		});
	}

	/**
	 * Attach people picker event listeners for an area
	 * @private
	 * @param {Object} area - Area object
	 */
	_attachPeoplePickerListeners(area) {
		const peoplePickerFields = ['executiveDirectors', 'allocators', 'contributors'];
		peoplePickerFields.forEach(fieldName => {
			this._addCleanupFunction(() => {
				const container = this.shadowRoot.getElementById(`pp-${fieldName}-${area.id}`);
				if (!container) return;
				this._initializePeoplePicker(container, area, fieldName);
				
			// Attach remove listeners for badge remove buttons
			Array.from(container.querySelectorAll('button.badge-remove')).forEach(btn => {
				const removeHandler = (ev) => {
					if(ev && ev.type==='keydown' && ev.key!=='Enter' && ev.key!==' ') return;
					const key = btn.getAttribute('data-key');
					const updated = (area[fieldName] || []).filter(p => this._getPersonKey(p) !== key);
					this.updateArea(area, { [fieldName]: updated });
				};
				btn.addEventListener('click', removeHandler);
				btn.addEventListener('keydown', removeHandler);
				this._cleanupFunctions.add(()=>{ btn.removeEventListener('click', removeHandler); btn.removeEventListener('keydown', removeHandler); });
			});
		});
	});
}

	/**
	 * Render a single area
	 * @private
	 * @param {Object} area - Area object to render
	 * @returns {string} HTML for the area
	 */
	_renderArea(area) {
		const displayIndex = area.index ?? area.order ?? 0;
		const isExpanded = this._expanded.has(area.id);
		const isEditable = this.canEdit();
		
		// Check if this is the last area (only last area can be deleted)
		const sortedAreas = this._getSortedAreas();
		const isLastArea = sortedAreas.length > 0 && sortedAreas[sortedAreas.length - 1].id === area.id;
		
		// Check if area name is selected (required for endorsement)
		const hasAreaName = !!(area.label && area.label.trim());

		const dropdownOptions = this._buildDropdownOptions(area);
		const missingFieldsHtml = this._renderMissingFields(area);
		const peopleChipsHtml = this._renderPeopleChips(area);
		
		// Endorsement indicator for heading
		const endorsedIndicator = area.endorsementComplete ? `<fluent-badge appearance="accent" style="margin-right: auto;">✓ Endorsed</fluent-badge>` : '';

		// Show area name in heading if selected
		const areaNameDisplay = hasAreaName ? `: ${this._escapeHtml(area.label)}` : '';

		return `
			<fluent-accordion-item id="panel-${area.id}" ${isExpanded ? 'expanded' : ''}>
				<div slot="heading" class="accordion-heading-content">
					<fluent-badge appearance="accent" circular>${displayIndex}</fluent-badge>
					<span>Supplementary Area ${displayIndex} Details${areaNameDisplay}</span>
					${endorsedIndicator}
				</div>
				${isEditable && isLastArea && this._model.areas.length > 1 ? `<button slot="end" type="button" id="del-${area.id}" class="delete-link" ${this._saving ? 'disabled' : ''} title="${this._saving ? 'Cannot delete while saving' : 'Delete this area'}">
					<svg width="16" height="16" viewBox="0 0 16 16" fill="currentColor"><path d="M11.5 3H14v1h-1v10a1 1 0 0 1-1 1H4a1 1 0 0 1-1-1V4H2V3h2.5V1.5A.5.5 0 0 1 5 1h6a.5.5 0 0 1 .5.5V3zM5 2v1h6V2H5zm7 2H4v9h8V4zM6 6h1v6H6V6zm3 0h1v6H9V6z"/></svg>
					Delete
				</button>` : ''}
				<div class="panel-body">
					${missingFieldsHtml}
					<div class="field-wrapper">
						<label>Supplementary Area Name</label>
						<fluent-select appearance="outline" id="name-${area.id}" ${!isEditable ? 'disabled' : ''} value="${this._escapeAttr(this._getAreaLabelValue(area))}">${dropdownOptions}</fluent-select>
					</div>
					<div class="field-wrapper">
						<label>Executive Directors</label>
						${peopleChipsHtml.executiveDirectors}
					</div>
					<div class="field-wrapper">
						<label>Allocators</label>
						${peopleChipsHtml.allocators}
					</div>
					<div class="field-wrapper">
						<label>Contributors</label>
						${peopleChipsHtml.contributors}
					</div>
					<div class="field-wrapper">
						<label>Comments</label>
						${this._renderCommentsSection(area, isEditable)}
					</div>
					<div class="field-wrapper">
						<label>Supplementary Area Endorsement</label>
						${this._renderField('endorsement', `<fluent-checkbox id="endorse-${area.id}" ${area.endorsementComplete ? 'checked' : ''} ${!isEditable || !hasAreaName ? 'disabled' : ''}>Mark Supplementary Area Endorsement as complete</fluent-checkbox>`)}
					</div>
				</div>
			</fluent-accordion-item>
		`;
	}

		/**
		 * Render field wrapper with metadata
		 * @private
		 * @param {Object|null} fieldMeta - Field metadata
		 * @param {string} innerHtml - Inner HTML content
		 * @returns {string} Wrapped HTML
		 */
		_renderField(fieldMeta, innerHtml) {
			if (!fieldMeta) return innerHtml;

			if (fieldMeta.isMissing) {
				return `<span style="color:#b00020;font-size:11px;">Field '${this._escapeHtml(fieldMeta.internalName)}' missing from content type</span>`;
			}

			return innerHtml;
		}

		// ========================================
		// PEOPLE PICKER METHODS
		// ========================================

		/**
		 * Initialize people picker functionality
		 * @private
		 * @param {HTMLElement} container - Container element
		 * @param {Object} area - Area object
		 * @param {string} fieldName - Field name to update
		 */
	_initializePeoplePicker(container, area, fieldName) {
		// Query for fluent-text-field with class people-search-input
		const input = container.querySelector('fluent-text-field.people-search-input');
		const resultsContainer = container.querySelector('.people-results');
	
			if (!input || !resultsContainer) return;

		// Accessibility semantics
		container.setAttribute('role','group');
		container.setAttribute('aria-label', fieldName.replace(/Ids$/,'').replace(/([A-Z])/g,' $1').trim());
		resultsContainer.setAttribute('role','listbox');
		resultsContainer.setAttribute('aria-multiselectable','true');
		if(!resultsContainer.id){ resultsContainer.id = `people-results-${fieldName}-${area.id}`; }
			input.setAttribute('role','combobox');
			input.setAttribute('aria-autocomplete','list');
			input.setAttribute('aria-expanded','false');
			input.setAttribute('aria-controls', resultsContainer.id);

			// Update the input field state based on user limit
			const currentCount = Array.isArray(area[fieldName]) ? area[fieldName].length : 0;
			const isAtLimit = currentCount >= APP_CONSTANTS.MAX_USERS_PER_FIELD;
			
			if (isAtLimit) {
				input.disabled = true;
				input.placeholder = `Maximum ${APP_CONSTANTS.MAX_USERS_PER_FIELD} users reached`;
				resultsContainer.innerHTML = '<div class="result-limit-notice">Maximum ' + APP_CONSTANTS.MAX_USERS_PER_FIELD + ' users reached</div>';
			} else {
				input.disabled = false;
				input.placeholder = 'Search people...';
			}

		let searchTimeout = null;
		let lastSearchTerm = '';
		let isSearching = false;
		let searchCounter = 0;

		const showListbox = () => {
			resultsContainer.hidden = false;
			input.setAttribute('aria-expanded', 'true');
		};

		const hideListbox = () => {
			resultsContainer.hidden = true;
			input.setAttribute('aria-expanded', 'false');
		};

		const performSearch = (searchTerm) => {
			if (isSearching) return; // Skip if search already in progress
			
			isSearching = true;
			const currentSearchId = ++searchCounter;
			
			// Track this as the latest search for this picker
			const pickerKey = `${fieldName}-${area.id}`;
			if (!this._latestSearchIds) this._latestSearchIds = {};
			this._latestSearchIds[pickerKey] = currentSearchId;
			
			// Show listbox with loading state
			resultsContainer.innerHTML = '<div class="result-loading" role="status" aria-live="polite">Searching…</div>';
			showListbox();
			input.setAttribute('aria-busy','true');
			
			this._searchPeople(searchTerm, resultsContainer, area, fieldName, currentSearchId)
				.finally(() => {
					// Only clear busy state if this is still the latest search
					if (currentSearchId === searchCounter) {
						input.removeAttribute('aria-busy');
						isSearching = false;
					}
				});
		};

		this._addCleanupFunction(() => {
			const inputHandler = (event) => {
				const currentCount = Array.isArray(area[fieldName]) ? area[fieldName].length : 0;
				if (currentCount >= APP_CONSTANTS.MAX_USERS_PER_FIELD) {
					input.disabled = true;
					input.placeholder = `Maximum ${APP_CONSTANTS.MAX_USERS_PER_FIELD} users reached`;
					return;
				}
				
				const searchTerm = event.target.value.trim();
				
				// Clear results if search term is too short
				if (searchTerm.length < 2) {
					resultsContainer.hidden = true;
					resultsContainer.innerHTML = '';
					input.setAttribute('aria-expanded','false');
					lastSearchTerm = '';
					if (searchTimeout) clearTimeout(searchTimeout);
					return;
				}
				
				// Skip if same as last search
				if (searchTerm === lastSearchTerm) return;
				
				lastSearchTerm = searchTerm;
				
				// Debounce the search
				if (searchTimeout) clearTimeout(searchTimeout);
				searchTimeout = setTimeout(() => {
					if (!isSearching) {
						performSearch(searchTerm);
					}
				}, APP_CONSTANTS.SEARCH_DEBOUNCE_DELAY);
			};
			input.addEventListener('input', inputHandler);
			return () => {
				if (searchTimeout) clearTimeout(searchTimeout);
				input.removeEventListener('input', inputHandler);
			};
		});

	this._addCleanupFunction(() => {
		const focusHandler = () => {
			// Update state
			const currentCount = Array.isArray(area[fieldName]) ? area[fieldName].length : 0;
			if (currentCount >= APP_CONSTANTS.MAX_USERS_PER_FIELD) {
				input.disabled = true;
				input.placeholder = `Maximum ${APP_CONSTANTS.MAX_USERS_PER_FIELD} users reached`;
				resultsContainer.innerHTML = '<div class="result-limit-notice">Maximum ' + APP_CONSTANTS.MAX_USERS_PER_FIELD + ' users reached</div>';
				showListbox();
			} else {
				input.disabled = false;
				input.placeholder = 'Type to search people...';
				
				// Show existing results or instruction
				if (resultsContainer.innerHTML.trim()) {
					showListbox();
				} else if (input.value.trim().length < 2) {
					resultsContainer.innerHTML = '<div class="result-empty">Type at least 2 characters to search</div>';
					showListbox();
				}
			}
		};
		
		const blurHandler = () => {
			// Hide listbox when input loses focus
			setTimeout(() => {
				// Check if we clicked inside the results
				if (!resultsContainer.contains(document.activeElement)) {
					hideListbox();
				}
			}, 150);
		};
		const keyHandler = (ev) => {
			const items = Array.from(resultsContainer.querySelectorAll('.result-item:not(.disabled)'));
			if(!items.length) return;
			let activeIndex = items.findIndex(i=> i.classList.contains('active'));
			if(ev.key==='ArrowDown'){ ev.preventDefault(); activeIndex = (activeIndex+1)%items.length; }
			else if(ev.key==='ArrowUp'){ ev.preventDefault(); activeIndex = (activeIndex<=0? items.length-1:activeIndex-1); }
			else if(ev.key==='Enter'){ if(activeIndex>=0){ items[activeIndex].click(); } return; }
			else if(ev.key==='Escape'){ input.blur(); return; }
			else { return; }
			items.forEach(i=> i.classList.remove('active'));
			if(activeIndex>=0){ items[activeIndex].classList.add('active'); }
		};
		
		input.addEventListener('focus', focusHandler);
		input.addEventListener('blur', blurHandler);
		input.addEventListener('keydown', keyHandler);
		return ()=> { 
			input.removeEventListener('focus', focusHandler);
			input.removeEventListener('blur', blurHandler);
			input.removeEventListener('keydown', keyHandler); 
		};
		});

	}

	/**
	 * Search for people
	 * @private
	 * @param {string} searchTerm - Term to search for
	 * @param {HTMLElement} resultsContainer - Container for results
	 * @param {Object} area - Area object
	 * @param {string} fieldName - Field name to update
	 * @param {number} searchId - Unique ID for this search request
	 * @returns {Promise<void>}
	 */
	async _searchPeople(searchTerm, resultsContainer, area, fieldName, searchId) {
		const currentCount = Array.isArray(area[fieldName]) ? area[fieldName].length : 0;
	if (currentCount >= APP_CONSTANTS.MAX_USERS_PER_FIELD) {
		resultsContainer.hidden = false;
		resultsContainer.innerHTML = '<div class="result-limit-notice">Maximum ' + APP_CONSTANTS.MAX_USERS_PER_FIELD + ' users reached</div>';
			return;
		}
		if (!searchTerm) {
			resultsContainer.hidden = true;
			resultsContainer.innerHTML = '';
			return;
		}

		try {
			const response = await api.invokeGet(undefined, {
				recordId: this._props.record.recordId,
				action: 'peopleSearch',
				q: searchTerm
			});

			// Check if this is still the latest search before rendering results
			const pickerKey = `${fieldName}-${area.id}`;
			if (!this._latestSearchIds) this._latestSearchIds = {};
			if (searchId !== this._latestSearchIds[pickerKey]) {
				// This is a stale search, ignore results
				return;
			}

			if (response.status !== 200) {
				resultsContainer.hidden = true;
				return;
			}

			const data = JSON.parse(response.body);
			const results = Array.isArray(data.results) ? data.results : [];

			this._renderPeopleSearchResults(results, resultsContainer, area, fieldName);
		} catch (error) {
			console.warn('People search failed:', error);
			resultsContainer.hidden = true;
		}
	}

		/**
		 * Render people search results
		 * @private
		 * @param {Array} results - Search results
		 * @param {HTMLElement} container - Container element
		 * @param {Object} area - Area object
		 * @param {string} fieldName - Field name to update
		 */
		_renderPeopleSearchResults(results, container, area, fieldName) {
		const norm = Array.isArray(results) ? results : [];
		const existingKeys = new Set((Array.isArray(area[fieldName]) ? area[fieldName] : []).map(p => this._getPersonKey(p)));
		const currentCount = Array.isArray(area[fieldName]) ? area[fieldName].length : 0;
		const isAtLimit = currentCount >= APP_CONSTANTS.MAX_USERS_PER_FIELD;
		const input = container.closest('.people-picker')?.querySelector('.people-search-input');
		if(input){ input.setAttribute('aria-expanded','true'); }
		const lastTerm = input ? input.value.trim() : '';
		const termRegex = lastTerm.length>=2 ? new RegExp(lastTerm.replace(/[.*+?^${}()|[\\]\\]/g,'\\$&'),'ig') : null;

		const resultsHtml = norm.length
			? norm.map((person, idx) => {
				const key = person.GraphUserId || person.graphUserId || person.UserPrincipalName || person.userPrincipalName || person.Mail || person.mail || person.id || '';
				const display = person.DisplayName || person.displayName || person.UserPrincipalName || person.userPrincipalName || person.Mail || person.mail || key;
				if(!key) return '';
				let encoded; try { encoded = encodeURIComponent(JSON.stringify(person)); } catch { return ''; }
			const isSelected = existingKeys.has(key);
			const isDisabled = isAtLimit && !isSelected;
			const highlighted = termRegex ? this._escapeHtml(display).replace(termRegex, m=>`<mark class=\"result-highlight\">${m}</mark>`) : this._escapeHtml(display);
			const metaText = isDisabled ? '<span class=\"result-meta\">(limit reached)</span>' : '';
			return `<div id=\"result-opt-${idx}\" role=\"option\" class=\"result-item${isSelected? ' selected' : ''}${isDisabled ? ' disabled' : ''}\" data-person=\"${encoded}\" data-key=\"${this._escapeAttr(key)}\" ${isSelected? 'aria-selected=\"true\"' : isDisabled ? 'aria-disabled=\"true\"' : ''}><span class=\"result-name\">${highlighted}</span>${metaText}</div>`;
			}).join('')
			: '<div class="result-empty">No matches found</div>';

		container.innerHTML = resultsHtml;
		if (isAtLimit) {
			container.innerHTML += '<div class="result-limit-notice">Maximum ' + APP_CONSTANTS.MAX_USERS_PER_FIELD + ' users reached</div>';
		}
		container.hidden = false;

		Array.from(container.querySelectorAll('.result-item:not(.selected):not(.disabled)')).forEach(element => {
				this._addCleanupFunction(() => {
					const clickHandler = () => {
						const encoded = element.getAttribute('data-person');
						const key = element.getAttribute('data-key');
						if(!encoded || !key) return;
						let raw = null; try { raw = JSON.parse(decodeURIComponent(encoded)); } catch { return; }
						if(existingKeys.has(key)) return; // duplicate guard

			// Store complete person object from search results
			const personObj = raw;
			const next = [...area[fieldName], personObj];
			this.updateArea(area, { [fieldName]: next });
			existingKeys.add(key); 
			element.classList.add('selected'); 
			element.setAttribute('aria-selected','true');
			
			// Blur input to close listbox after selection
			const pickerInput = container.closest('.people-picker')?.querySelector('.people-search-input');
			if (pickerInput) {
				setTimeout(() => pickerInput.blur(), 100);
			}
		};
		element.addEventListener('click', clickHandler);
		return () => element.removeEventListener('click', clickHandler);
				});
			});
		}

		// ========================================
		// UTILITY METHODS
		// ========================================

		/**
		 * Get sorted areas by index
		 * @private
		 * @returns {Array} Sorted areas array
		 */
		_getSortedAreas() {
			if (!this._model.areas) return [];

			return [...this._model.areas].sort((areaA, areaB) => {
				const indexA = (typeof areaA.index === 'number' ? areaA.index : (typeof areaA.order === 'number' ? areaA.order : 0));
				const indexB = (typeof areaB.index === 'number' ? areaB.index : (typeof areaB.order === 'number' ? areaB.order : 0));
				return indexA - indexB;
			});
		}

	/**
	 * Update area heading to show/hide endorsement indicator and area name without full re-render
	 * @private
	 * @param {Object} area - Area object
	 */
	_updateAreaHeading(area) {
		const panel = this.shadowRoot?.getElementById(`panel-${area.id}`);
		if (!panel) return;

		const headingContent = panel.querySelector('.accordion-heading-content');
		if (!headingContent) return;

		// Update the heading text to include/exclude area name
		const displayIndex = area.index ?? area.order ?? 0;
		const hasAreaName = !!(area.label && area.label.trim());
		const areaNameDisplay = hasAreaName ? `: ${this._escapeHtml(area.label)}` : '';
		const headingSpan = headingContent.querySelector('span:not([class])');
		if (headingSpan) {
			headingSpan.textContent = `Supplementary Area ${displayIndex} Details${areaNameDisplay}`;
		}

		// Remove existing endorsement badge if present
		const existingBadge = headingContent.querySelector('fluent-badge[style*="margin-right: auto"]');
		if (existingBadge) {
			existingBadge.remove();
		}

		// Add new endorsement badge if endorsed
		if (area.endorsementComplete) {
			const badge = document.createElement('fluent-badge');
			badge.setAttribute('appearance', 'accent');
			badge.style.marginRight = 'auto';
			badge.textContent = '✓ Endorsed';
			headingContent.appendChild(badge);
		}
	}

		/**
		 * Get the value (id) for the area's label to set on fluent-select
		 * @private
		 * @param {Object} area - Area object
		 * @returns {string} The value/id corresponding to the label
		 */
		_getAreaLabelValue(area) {
			if (!area.label) return '';
			const option = (this._dropdownOptions || []).find(opt => 
				(opt.label === area.label) || (opt.id === area.label) || (opt.value === area.label)
			);
			return option ? (option.id || option.value || '') : '';
		}

		/**
		 * Build dropdown options HTML
		 * @private
		 * @param {Object} area - Area object
		 * @returns {string} Options HTML
		 */
		_buildDropdownOptions(area) {
			const options = Array.isArray(this._dropdownOptions) ? this._dropdownOptions : [];
			const optionValues = options.map(option => ({
				id: (option.id || option.value || ''),
				label: option.label || option.id || option.value || ''
			}));
	
			return [''] // Empty option for "Select"
				.concat(optionValues.map(opt => opt.id))
				.map(value => {
					const foundOption = optionValues.find(opt => opt.id === value);
					const label = foundOption ? foundOption.label : value;
	
					return `<fluent-option value="${this._escapeAttr(value)}">${value ? this._escapeHtml(label) : '-- Select --'}</fluent-option>`;
				})
				.join('') +
				(area.label && !optionValues.some(opt => opt.label === area.label) ?
					`<span class="stale-indicator" aria-label="Stale value" title="This value no longer exists in configuration; it will be preserved until changed.">⚠</span>` : '');
		}

		/**
	 * Render global missing fields notice
	 * @private
	 * @param {Array<string>} globalMissing - Array of missing field internal names
	 * @returns {string} Missing fields HTML
	 */
	_renderGlobalMissingFields(globalMissing) {
		// Filter out due date from global display (shown inline with field)
		const filtered = globalMissing.filter(field => field !== this._dueDateFieldInternalName);
		
		if (filtered.length === 0) return '';
		
		return `<fluent-card class="notification-card notification-warning">
			<div class="notification-content">
				<span class="notification-icon">⚠</span>
				<div class="notification-body">
					<div class="notification-title">Missing Configuration</div>
					<div class="notification-text">
						The following fields are not configured in the content type and cannot be synchronized to the record:
					</div>
					<div class="notification-fields">
						${filtered.map(field => `<code>${this._escapeHtml(field)}</code>`).join(' ')}
					</div>
					<div class="notification-help">
						Ask an administrator to add these fields to the content type. Data will still be saved in plugin storage.
					</div>
				</div>
			</div>
		</fluent-card>`;
	}

	/**
	 * Render missing fields notice (per-area, now deprecated - returns empty)
	 * @private
	 * @param {Object} area - Area object
	 * @returns {string} Missing fields HTML
	 */
	_renderMissingFields(area) {
		// Per-area missing fields no longer used (we have global aggregated fields now)
		return '';
	}

	/**
	 * Render comments section as conversation thread
	 * @private
	 * @param {Object} area - Area object
	 * @param {boolean} isEditable - Whether user can edit
	 * @returns {string} Comments section HTML
	 */
	_renderCommentsSection(area, isEditable) {
		const comments = Array.isArray(area.comments) ? area.comments : [];
		
		// Render existing comments
		const commentsHtml = comments.length > 0 
			? comments.map(comment => this._renderComment(comment)).join('')
			: '<div class="comments-empty">No comments yet</div>';

		// Render new comment input (only if editable)
		const newCommentInput = isEditable 
			? `<fluent-text-area 
					class="comment-input" 
					id="new-comment-${area.id}" 
					placeholder="Type a comment here... (will be added on Save)" 
					rows="3" 
					resize="vertical"
					appearance="outline"></fluent-text-area>`
			: '';

		return `<div class="comments-section">
			<div class="comments-thread">${commentsHtml}</div>
			${newCommentInput}
		</div>`;
	}

	/**
	 * Render a single comment
	 * @private
	 * @param {Object} comment - Comment object
	 * @returns {string} Comment HTML
	 */
	_renderComment(comment) {
		
		const date = comment.createdUtc || new Date().toISOString();
		const formattedDate = this._formatCommentDate(date);
		const text = comment.text || '';
		const author = comment.author?.displayName || '';

		return `<div class="comment-item">
			<div class="comment-header">
				<span class="comment-author">${this._escapeHtml(author)}</span>
				<span class="comment-date">${this._escapeHtml(formattedDate)}</span>
			</div>
			<div class="comment-text">${this._escapeHtml(text)}</div>
		</div>`;
	}

	/**
	 * Format comment date for display
	 * @private
	 * @param {string} dateString - ISO date string
	 * @returns {string} Formatted date
	 */
	_formatCommentDate(dateString) {
		try {
			const date = new Date(dateString);
			const now = new Date();
			const diff = now - date;
			const hours = Math.floor(diff / (1000 * 60 * 60));
			const days = Math.floor(hours / 24);

			if (hours < 1) return 'Just now';
			if (hours < 24) return `${hours}h ago`;
			if (days < 7) return `${days}d ago`;
			
			// Format as date
			return date.toLocaleDateString(undefined, { month: 'short', day: 'numeric', year: 'numeric' });
		} catch {
			return dateString;
		}
	}

		/**
		 * Render people chips for an area
		 * @private
		 * @param {Object} area - Area object
		 * @returns {Object} Object with HTML for each people picker field
		 */
	_renderPeopleChips(area) {
		const fields = ['executiveDirectors', 'allocators', 'contributors'];
		return fields.reduce((acc, fieldName) => {
			const people = Array.isArray(area[fieldName]) ? area[fieldName] : [];
			const fieldMeta = this._getFieldMetadata(fieldName, area);
			const isAtLimit = people.length >= APP_CONSTANTS.MAX_USERS_PER_FIELD;
			
	// Render selected users as Fluent UI badges with delete button inside
	const badgesHtml = people.map(p => {
		const key = this._getPersonKey(p);
		const label = this._getPersonDisplay(p);
		return `<fluent-badge appearance="accent" class="person-badge" data-person-key="${this._escapeAttr(key)}">
			${this._escapeHtml(label)}
			${this.canEdit() ? `<button class="badge-remove" data-key="${this._escapeAttr(key)}" aria-label="Remove ${this._escapeAttr(label)}" title="Remove ${this._escapeAttr(label)}" type="button">×</button>` : ''}
		</fluent-badge>`;
	}).join('');
		
		const countIndicator = `<span class="people-count" aria-label="${people.length} of ${APP_CONSTANTS.MAX_USERS_PER_FIELD} selected">${people.length}/${APP_CONSTANTS.MAX_USERS_PER_FIELD}</span>`;
		
		let inputHtml = '';
		if(this.canEdit()){
			const placeholder = isAtLimit ? `Maximum ${APP_CONSTANTS.MAX_USERS_PER_FIELD} users reached` : 'Type to search people...';
			inputHtml = `<div class="people-input-wrapper">
				<fluent-text-field class="people-search-input" placeholder="${this._escapeAttr(placeholder)}" ${isAtLimit ? 'disabled' : ''} aria-label="Search people" appearance="outline"></fluent-text-field>
				${countIndicator}
				<div class="people-results" hidden></div>
			</div>`;
		}
		
		const pickerHtml = `<div class="people-picker${isAtLimit ? ' at-limit' : ''}" id="pp-${this._escapeAttr(fieldName)}-${this._escapeAttr(area.id)}" data-field="${this._escapeAttr(fieldName)}">
			${inputHtml}
			<div class="people-badges-container">${badgesHtml}</div>
		</div>`;
			
			acc[fieldName] = this._renderField(fieldMeta, pickerHtml);
			return acc;
		}, {});
	}

		/**
	 * Get field metadata for a people picker field (no longer needed with aggregated fields)
		 * @private
		 * @param {string} fieldName - Name of the field
		 * @param {Object} area - Area object
		 * @returns {Object|null} Field metadata
		 */
		_getFieldMetadata(fieldName, area) {
		// With aggregated fields, we no longer have per-area field internal names
		// Missing field status is now tracked globally and displayed at the top
		return null;
		}

		/**
		 * Normalize area indices to ensure consistency
		 * @private
		 */
		_normalizeAreaIndices() {
			if (!this._model || !Array.isArray(this._model.areas)) return;

			let hasChanged = false;
			this._model.areas.forEach((area, index) => {
				if (typeof area.index !== 'number') {
					area.index = typeof area.order === 'number' ? area.order : index + 1;
					hasChanged = true;
				}
			});

			// Note: Changed areas will be persisted on next structural action
		}

	/**
	 * Mark the current state as dirty (has unsaved changes)
	 * @private
	 */
	_markAsDirty() {
		const currentHash = this._computeValuesHash();
		const wasDirty = this._dirtyValues;
		if (this._baselineValuesHash && currentHash === this._baselineValuesHash) {
			this._dirtyValues = false;
		} else {
			this._dirtyValues = true;
			
			// Clear success message when user makes changes after save
			// This prevents stale "All changes saved" message from persisting
			if (this._flashMessage && this._flashMessage.type === 'success') {
				this._flashMessage = null;
				if (this._flashTimer) {
					clearTimeout(this._flashTimer);
					this._flashTimer = null;
				}
				
				// Manually remove the flash message element from DOM without full re-render
				const flashElement = this.shadowRoot?.querySelector('.notification-flash');
				if (flashElement) {
					flashElement.remove();
				}
			}
		}
		
		// Update Save button state without full re-render
		if (wasDirty !== this._dirtyValues) {
			const saveBtn = this.shadowRoot?.getElementById('saveBtn');
			if (saveBtn) {
				saveBtn.disabled = !this._dirtyValues || this._saving;
			}
		}
	}

		/**
		 * Update the baseline hash for change tracking
		 * @private
		 */
		_updateBaselineHash() {
			this._baselineValuesHash = this._computeValuesHash();
		}

	/**
	 * Compute hash of current values for change tracking
	 * @private
	 * @returns {string} Hash string
	 */
	_computeValuesHash() {
		if (!this._model) return 'no-model';

		const normalizePeople = (arr) => {
			return (arr || []).map(p => {
				if (typeof p === 'string') return { k: p };
				if (p && typeof p === 'object') {
					const k = p.GraphUserId || p.graphUserId || p.UserPrincipalName || p.userPrincipalName || p.Mail || p.mail || p.id || p.DisplayName || p.displayName || '';
					return {
						k,
						GraphUserId: p.GraphUserId || p.graphUserId,
						DisplayName: p.DisplayName || p.displayName,
						JobTitle: p.JobTitle || p.jobTitle,
						Mail: p.Mail || p.mail,
						UserPrincipalName: p.UserPrincipalName || p.userPrincipalName
					};
				}
				return { k: String(p) };
			}).sort((a,b)=> (a.k||'').localeCompare(b.k||''));
		};

		const normalizeComments = (arr) => {
			return (arr || []).map(c => ({
				id: c.id,
				text: c.text,
				createdUtc: c.createdUtc,
				modifiedUtc: c.modifiedUtc,
				author: c.author
			})).sort((a,b)=> (a.createdUtc||'').localeCompare(b.createdUtc||''));
		};

		const normalizedModel = {
			dueDate: this._model.dueDate || null,
			areas: (this._model.areas || []).map(area => {
				const commentInput = this.shadowRoot?.getElementById(`new-comment-${area.id}`);
				const newCommentText = commentInput ? (commentInput.currentValue || commentInput.value || '').trim() : null;
				
				const commentsArray = Array.isArray(area.comments) ? area.comments : [];

				return {
					id: area.id,
					label: area.label || null,
					executiveDirectors: normalizePeople(area.executiveDirectors),
					allocators: normalizePeople(area.allocators),
					contributors: normalizePeople(area.contributors),
					comments: normalizeComments(commentsArray),
					newComment: newCommentText || null,
					endorsementComplete: !!area.endorsementComplete
				};
			})
		};

		try {
			const jsonString = JSON.stringify(normalizedModel);
			let hash = 0;
			for (let i = 0; i < jsonString.length; i++) {
				hash = (hash * 31 + jsonString.charCodeAt(i)) >>> 0;
			}
			return hash.toString(16);
		} catch (error) {
			return Math.random().toString();
		}
	}

		/**
		 * Debounce function calls
		 * @private
		 * @param {Function} func - Function to debounce
		 * @param {number} wait - Wait time in milliseconds
		 * @returns {Function} Debounced function
		 */
		_debounce(func, wait) {
			let timeout;
			return function executedFunction(...args) {
				const later = () => {
					clearTimeout(timeout);
					func.apply(this, args);
				};
				clearTimeout(timeout);
				timeout = setTimeout(later, wait);
			};
		}

		/**
		 * Add a cleanup function for proper memory management
		 * @private
		 * @param {Function} cleanupFn - Cleanup function that returns a cleanup function
		 */
		_addCleanupFunction(cleanupFn) {
			try {
				const actualCleanup = cleanupFn();
				if(typeof actualCleanup === 'function') this._cleanupFunctions.add(actualCleanup);
			} catch (error) {
				console.warn('Failed to add cleanup function:', error);
			}
		}

		/**
		 * Clean up event listeners and timers
		 * @private
		 */
		_cleanup() {
			this._cleanupFunctions.forEach(cleanup => {
				try {
					cleanup();
				} catch (error) {
					console.warn('Cleanup function failed:', error);
				}
			});
			this._cleanupFunctions.clear();

			if (this._flashTimer) {
				clearTimeout(this._flashTimer);
				this._flashTimer = null;
			}
		}

		/**
		 * Handle errors consistently
		 * @private
		 * @param {string} message - Error message
		 * @param {Error} [originalError] - Original error object
		 */
		_handleError(message, originalError) {
			console.error(message, originalError);
			this._error = originalError ? `${message}: ${originalError.message}` : message;
			this._notify(message, NOTIFICATION_TYPES.ERROR);
		}

		/**
		 * Escape HTML characters
		 * @private
		 * @param {string} text - Text to escape
		 * @returns {string} Escaped text
		 */
		_escapeHtml(text) {
			return String(text)
				.replace(/&/g, '&amp;')
				.replace(/</g, '&lt;')
				.replace(/>/g, '&gt;');
		}

		/**
		 * Escape attribute values
		 * @private
		 * @param {string} text - Text to escape
		 * @returns {string} Escaped text
		 */
		_escapeAttr(text) {
			return String(text)
				.replace(/&/g,'&amp;')
				.replace(/</g,'&lt;')
				.replace(/>/g,'&gt;')
				.replace(/"/g,'&quot;');
		}

	// ========================================
	// PEOPLE UTILITY HELPERS (for storing objects instead of IDs)
	// ========================================
	_getPersonKey(person) {
		if(!person) return '';
		if (typeof person === 'string') return person;
		return person.GraphUserId || person.graphUserId || person.UserPrincipalName || person.userPrincipalName || person.Mail || person.mail || person.id || '';
	}
	_getPersonDisplay(person) {
		if(!person) return '';
		if (typeof person === 'string') return person;
		return person.DisplayName || person.displayName || person.UserPrincipalName || person.userPrincipalName || person.Mail || person.mail || person.id || 'Unknown';
	}

	// ========================================
	// DATE UTILITY HELPERS
	// ========================================
	/**
	 * Extract date part from ISO 8601 DateTime string for HTML date input
	 * Converts "2025-10-16T00:00:00Z" to "2025-10-16"
	 * @param {string|null} dateTimeString - ISO 8601 DateTime string or simple date string
	 * @returns {string} Date in YYYY-MM-DD format or empty string
	 */
	_extractDatePart(dateTimeString) {
		if (!dateTimeString) return '';
		
		// If already in simple date format (YYYY-MM-DD), return as-is
		if (/^\d{4}-\d{2}-\d{2}$/.test(dateTimeString)) {
			return dateTimeString;
		}
		
		// Extract date part from ISO 8601 format (2025-10-16T00:00:00Z -> 2025-10-16)
		const match = dateTimeString.match(/^(\d{4}-\d{2}-\d{2})/);
		return match ? match[1] : '';
	}

	/**
	 * Extract time part from ISO 8601 DateTime string
	 * Converts "2025-10-16T14:30:00Z" to "14:30"
	 * @param {string|null} dateTimeString - ISO 8601 DateTime string
	 * @returns {string} Time in HH:MM format or empty string
	 */
	_extractTimePart(dateTimeString) {
		if (!dateTimeString) return '';
		
		// Extract time part from ISO 8601 format
		const match = dateTimeString.match(/T(\d{2}:\d{2})/);
		return match ? match[1] : '';
	}

	/**
	 * Generate time options from 7:00 AM to 7:00 PM
	 * @returns {Array<{value: string, label: string}>} Array of time options
	 */
	_generateTimeOptions() {
		const options = [];
		// 7:00 AM = 07:00, 7:00 PM = 19:00
		for (let hour = 7; hour <= 19; hour++) {
			const hour24 = hour.toString().padStart(2, '0');
			const hour12 = hour > 12 ? hour - 12 : (hour === 0 ? 12 : hour);
			const ampm = hour >= 12 ? 'PM' : 'AM';
			const label = `${hour12}:00 ${ampm}`;
			options.push({ value: `${hour24}:00`, label });
		}
		return options;
	}

	/**
	 * Render time dropdown options HTML
	 * @param {string} selectedTime - Currently selected time in HH:MM format
	 * @returns {string} Options HTML
	 */
	_renderTimeOptions(selectedTime) {
		const options = this._generateTimeOptions();
		const emptyOption = `<fluent-option value="">-- Select time --</fluent-option>`;
		const optionsHtml = options.map(opt => 
			`<fluent-option value="${this._escapeAttr(opt.value)}">${this._escapeHtml(opt.label)}</fluent-option>`
		).join('');
		return emptyOption + optionsHtml;
	}

	/**
	 * Combine date and time into ISO 8601 DateTime string
	 * @param {string} datePart - Date in YYYY-MM-DD format
	 * @param {string} timePart - Time in HH:MM format
	 * @returns {string|null} Combined DateTime or null if no date
	 */
	_combineDateAndTime(datePart, timePart) {
		if (!datePart) return null;
		if (!timePart) return datePart; // Return just date if no time selected
		return `${datePart}T${timePart}:00`;
	}
	}

	function escapeHtml(s){ return String(s).replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;'); }
	function escapeAttr(s){ return String(s).replace(/"/g,'&quot;'); }

	if(!customElements.get('supplementary-areas-tab')){
		customElements.define('supplementary-areas-tab', SupplementaryAreasTab);
	}

	window.customTabs = window.customTabs || [];
	window.customTabs.push(function(record){
		if(!record || !record.recordId) return null;
		const allowedTypes = ['Cabinet or Committee Briefing (CabBRI)','Coordination Comment Briefing (CabCRD)'];
		if(!allowedTypes.includes(record.recordType)) return null;
		return [{
			tabName: 'Supplementary Areas',
			order: 60,
			webComponentTag: 'supplementary-areas-tab',
			props: { record }
		}];
	});
})(api,pid);

